// Kurs LinkedIn Content Script
// Adds "Gem til Kurs" button on LinkedIn job pages
// and scrapes LinkedIn feed posts

const BUTTON_CLASS = 'kurs-save-btn';
const FEED_BUTTON_CLASS = 'kurs-feed-btn';

// Track current job ID for detecting job changes
let lastJobId = '';

/**
 * Extract job ID from URL (currentJobId parameter or /view/ path)
 */
function getJobIdFromUrl() {
  const url = new URL(window.location.href);

  // Try currentJobId parameter first (search results page)
  const currentJobId = url.searchParams.get('currentJobId');
  if (currentJobId) {
    return currentJobId;
  }

  // Try /jobs/view/ID/ path (direct job page)
  const viewMatch = url.pathname.match(/\/jobs\/view\/(\d+)/);
  if (viewMatch) {
    return viewMatch[1];
  }

  return null;
}

// ========================================
// API HELPER - Uses background service worker
// ========================================

/**
 * Send API request via background service worker
 * This avoids Chrome's security restrictions on localhost requests
 */
async function apiRequest(endpoint, method = 'GET', body = null) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage(
      { action: 'fetch', endpoint, method, body },
      (response) => {
        if (chrome.runtime.lastError) {
          resolve({ error: chrome.runtime.lastError.message });
        } else {
          resolve(response || { error: 'No response' });
        }
      }
    );
  });
}

// ========================================
// JOB PAGE FUNCTIONALITY
// ========================================

/**
 * Scrape job data from the current LinkedIn job page
 */
function scrapeJobData() {
  // Job title - try multiple selectors for different LinkedIn layouts
  const titleSelectors = [
    '.job-details-jobs-unified-top-card__job-title',
    '.jobs-unified-top-card__job-title',
    '.top-card-layout__title',
    'h1.t-24',
    'h1'
  ];

  let jobTitle = '';
  for (const selector of titleSelectors) {
    const el = document.querySelector(selector);
    if (el && el.textContent.trim()) {
      jobTitle = el.textContent.trim();
      break;
    }
  }

  // Company name - try multiple selectors
  const companySelectors = [
    '.job-details-jobs-unified-top-card__company-name',
    '.jobs-unified-top-card__company-name',
    '.top-card-layout__second-subline a',
    '.topcard__org-name-link',
    '[data-tracking-control-name="public_jobs_topcard-org-name"]'
  ];

  let companyName = '';
  let companyLink = '';
  for (const selector of companySelectors) {
    const el = document.querySelector(selector);
    if (el && el.textContent.trim()) {
      companyName = el.textContent.trim();
      // Try to get company LinkedIn URL
      const linkEl = el.tagName === 'A' ? el : el.querySelector('a');
      if (linkEl && linkEl.href) {
        companyLink = linkEl.href;
      }
      break;
    }
  }

  // Job description - try multiple selectors
  const descSelectors = [
    '.jobs-description-content__text',
    '.jobs-box__html-content',
    '.description__text',
    '.show-more-less-html__markup'
  ];

  let description = '';
  for (const selector of descSelectors) {
    const el = document.querySelector(selector);
    if (el && el.textContent.trim()) {
      description = el.textContent.trim();
      break;
    }
  }

  // Current URL
  const jobUrl = window.location.href;

  return {
    company_name: companyName || 'Ukendt virksomhed',
    company_linkedin_url: companyLink,
    job_posting_text: jobTitle ? `${jobTitle}\n\n${description}` : description,
    source_url: jobUrl
  };
}

/**
 * Check if job is already saved (by job ID for reliable matching)
 */
async function checkIfJobSaved(url, jobId) {
  // Use job ID for matching (more reliable than full URL)
  let endpoint = `/api/entry/exists?url=${encodeURIComponent(url)}`;
  if (jobId) {
    endpoint += `&jobId=${encodeURIComponent(jobId)}`;
  }
  const result = await apiRequest(endpoint);
  if (result.success && result.data) {
    return result.data.exists;
  }
  return false;
}

/**
 * Create and return the save button element
 */
function createSaveButton(alreadySaved = false) {
  const button = document.createElement('button');
  button.className = BUTTON_CLASS;

  if (alreadySaved) {
    button.innerHTML = '<span class="kurs-icon">&#10003;</span> Allerede gemt';
    button.classList.add('kurs-already-saved');
    button.disabled = true;
  } else {
    button.innerHTML = '<span class="kurs-icon">&#128203;</span> Gem til Kurs';
    button.addEventListener('click', handleSaveClick);
  }

  return button;
}

/**
 * Handle save button click
 */
async function handleSaveClick(event) {
  event.preventDefault();
  event.stopPropagation();

  const button = event.currentTarget;

  // Prevent double-clicks
  if (button.disabled) return;

  button.disabled = true;
  button.innerHTML = '<span class="kurs-icon">&#8987;</span> Gemmer...';
  button.classList.add('kurs-saving');

  try {
    const data = scrapeJobData();
    const result = await apiRequest('/api/entry', 'POST', data);

    if (result.success && result.data) {
      if (result.data.already_existed) {
        button.innerHTML = '<span class="kurs-icon">&#10003;</span> Allerede gemt';
        button.classList.remove('kurs-saving');
        button.classList.add('kurs-already-saved');
      } else {
        button.innerHTML = '<span class="kurs-icon">&#10003;</span> Gemt!';
        button.classList.remove('kurs-saving');
        button.classList.add('kurs-success');

        setTimeout(() => {
          button.innerHTML = '<span class="kurs-icon">&#10003;</span> Allerede gemt';
          button.classList.remove('kurs-success');
          button.classList.add('kurs-already-saved');
        }, 2000);
      }
    } else {
      throw new Error(result.error || 'Unknown error');
    }

  } catch (error) {
    button.innerHTML = '<span class="kurs-icon">&#10060;</span> Kurs kører ikke';
    button.classList.remove('kurs-saving');
    button.classList.add('kurs-error');
    button.disabled = false;

    setTimeout(() => {
      button.innerHTML = '<span class="kurs-icon">&#128203;</span> Gem til Kurs';
      button.classList.remove('kurs-error');
    }, 3000);
  }
}

/**
 * Find the best container to insert the button
 */
function findButtonContainer() {
  // Try various containers where the button fits well
  const containerSelectors = [
    '.job-details-jobs-unified-top-card__job-title',
    '.jobs-unified-top-card__job-title',
    '.top-card-layout__title',
    '.jobs-details-top-card__content-container',
    '.job-view-layout',
    '.jobs-search__job-details--container'
  ];

  for (const selector of containerSelectors) {
    const container = document.querySelector(selector);
    if (container) {
      return container;
    }
  }

  return null;
}

/**
 * Update button to "already saved" state
 */
function markButtonAsSaved(button) {
  button.innerHTML = '<span class="kurs-icon">&#10003;</span> Allerede gemt';
  button.classList.add('kurs-already-saved');
  button.disabled = true;
  // Remove click listener by cloning
  const newButton = button.cloneNode(true);
  button.parentNode.replaceChild(newButton, button);
}

/**
 * Initialize the extension on job pages
 */
async function initJobPage() {
  const currentJobId = getJobIdFromUrl();

  // No job ID found - not viewing a specific job
  if (!currentJobId) {
    return;
  }

  // Check if job ID changed (user clicked different job)
  if (currentJobId !== lastJobId) {
    // Job changed - remove old button and create new one
    cleanupJobButtons();
    lastJobId = currentJobId;
  }

  // Don't add button if it already exists for this job
  if (document.querySelector(`.${BUTTON_CLASS}`)) {
    return;
  }

  const container = findButtonContainer();
  if (!container) {
    return;
  }

  // Create button immediately (not saved state)
  const button = createSaveButton(false);

  // Insert button after the container element
  if (container.parentNode) {
    container.parentNode.insertBefore(button, container.nextSibling);
  }

  // Check if already saved in the background, then update button if needed
  const currentUrl = window.location.href;
  try {
    const alreadySaved = await checkIfJobSaved(currentUrl, currentJobId);
    if (alreadySaved) {
      const existingButton = document.querySelector(`.${BUTTON_CLASS}`);
      if (existingButton) {
        markButtonAsSaved(existingButton);
      }
    }
  } catch (e) {
    // Ignore errors - button is already shown
    console.log('[Kurs] Could not check if job is saved:', e);
  }
}

/**
 * Clean up old job buttons when navigating between jobs
 */
function cleanupJobButtons() {
  const oldButtons = document.querySelectorAll(`.${BUTTON_CLASS}`);
  oldButtons.forEach(btn => btn.remove());
}

/**
 * Clean up feed buttons and reset processed flags
 */
function cleanupFeedButtons() {
  const oldButtons = document.querySelectorAll(`.${FEED_BUTTON_CLASS}`);
  oldButtons.forEach(btn => btn.remove());

  // Reset processed flags so posts can be re-checked
  const processedPosts = document.querySelectorAll('[data-kurs-processed]');
  processedPosts.forEach(post => delete post.dataset.kursProcessed);
}

// ========================================
// FEED FUNCTIONALITY
// ========================================

/**
 * Scrape post data from a feed post element
 */
function scrapePostData(postElement) {
  // Author name - look for span with aria-hidden inside actor links
  const authorNameSelectors = [
    '.update-components-actor__title span[aria-hidden="true"]',
    '.feed-shared-actor__name span[aria-hidden="true"]',
    '.update-components-actor__name span[aria-hidden="true"]',
    '.update-components-actor__title > span > span[aria-hidden="true"]',
    '.feed-shared-actor__title span[aria-hidden="true"]',
    '.update-components-actor__title',
    '.feed-shared-actor__name',
    '.update-components-actor__name'
  ];
  let authorName = '';
  for (const selector of authorNameSelectors) {
    const el = postElement.querySelector(selector);
    if (el && el.textContent.trim()) {
      let name = el.textContent.trim().split('\n')[0].trim();
      // Skip if it looks like follower count or is empty
      if (!name.includes('followers') && !name.includes('follower') && name.length > 0 && name.length < 100) {
        authorName = name;
        break;
      }
    }
  }

  // Author title/headline
  const authorTitleSelectors = [
    '.feed-shared-actor__description',
    '.update-components-actor__description',
    '.feed-shared-actor__sub-description',
    '[data-anonymize="headline"]'
  ];
  let authorTitle = '';
  for (const selector of authorTitleSelectors) {
    const el = postElement.querySelector(selector);
    if (el && el.textContent.trim()) {
      authorTitle = el.textContent.trim().split('\n')[0].trim();
      break;
    }
  }

  // Author LinkedIn URL
  const authorLinkSelectors = [
    '.feed-shared-actor__container-link',
    '.update-components-actor__container-link',
    '.feed-shared-actor__meta-link',
    'a[data-control-name="actor"]'
  ];
  let authorLinkedInUrl = '';
  for (const selector of authorLinkSelectors) {
    const el = postElement.querySelector(selector);
    if (el && el.href) {
      authorLinkedInUrl = el.href.split('?')[0];
      break;
    }
  }

  // Post content - look for the text content
  const contentSelectors = [
    '.feed-shared-update-v2__description',
    '.update-components-text',
    '.feed-shared-text',
    '.feed-shared-inline-show-more-text',
    '[data-test-id="main-feed-activity-card__commentary"]',
    '.break-words'
  ];
  let content = '';
  for (const selector of contentSelectors) {
    const el = postElement.querySelector(selector);
    if (el && el.textContent.trim()) {
      content = el.textContent.trim();
      break;
    }
  }

  // Post URL - try multiple methods
  let postUrl = '';

  // Method 1: data-urn attribute
  const activityEl = postElement.querySelector('[data-urn]');
  if (activityEl && activityEl.dataset.urn) {
    const urn = activityEl.dataset.urn;
    const match = urn.match(/activity:(\d+)/);
    if (match) {
      postUrl = `https://www.linkedin.com/feed/update/urn:li:activity:${match[1]}`;
    }
  }

  // Method 2: data-id attribute on the post
  if (!postUrl) {
    const dataId = postElement.getAttribute('data-id');
    if (dataId) {
      const match = dataId.match(/activity:(\d+)/);
      if (match) {
        postUrl = `https://www.linkedin.com/feed/update/urn:li:activity:${match[1]}`;
      }
    }
  }

  // Method 3: Link in the post
  if (!postUrl) {
    const shareLink = postElement.querySelector('a[href*="/feed/update/"]');
    if (shareLink) {
      postUrl = shareLink.href.split('?')[0];
    }
  }

  // Method 4: Time/date link often links to the post
  if (!postUrl) {
    const timeLink = postElement.querySelector('.feed-shared-actor__sub-description a, a.app-aware-link[href*="activity"]');
    if (timeLink && timeLink.href) {
      postUrl = timeLink.href.split('?')[0];
    }
  }

  // Method 5: Look for any link containing activity ID
  if (!postUrl) {
    const allLinks = postElement.querySelectorAll('a[href*="urn:li:activity"]');
    for (const link of allLinks) {
      const match = link.href.match(/urn:li:activity:(\d+)/);
      if (match) {
        postUrl = `https://www.linkedin.com/feed/update/urn:li:activity:${match[1]}`;
        break;
      }
    }
  }

  // Method 6: Check parent element for data-urn
  if (!postUrl) {
    let parent = postElement.parentElement;
    while (parent && !postUrl) {
      if (parent.dataset && parent.dataset.urn) {
        const match = parent.dataset.urn.match(/activity:(\d+)/);
        if (match) {
          postUrl = `https://www.linkedin.com/feed/update/urn:li:activity:${match[1]}`;
        }
      }
      parent = parent.parentElement;
    }
  }

  // Method 7: Generate a pseudo-URL based on author + content hash if no URL found
  // This allows saving posts even without finding LinkedIn's activity ID
  if (!postUrl && authorName && content) {
    const hash = (authorName + content.substring(0, 100)).split('').reduce((a, b) => {
      a = ((a << 5) - a) + b.charCodeAt(0);
      return a & a;
    }, 0);
    postUrl = `https://www.linkedin.com/feed/post-${Math.abs(hash)}`;
  }

  console.log('[Kurs] Scraped post:', { authorName, authorTitle, content: content.substring(0, 50), postUrl });

  return {
    author_name: authorName,
    author_title: authorTitle,
    author_linkedin_url: authorLinkedInUrl,
    content: content,
    post_url: postUrl
  };
}

/**
 * Check if post is already saved
 */
async function checkIfPostSaved(url) {
  if (!url) return { exists: false };

  const result = await apiRequest(`/api/post/exists?url=${encodeURIComponent(url)}`);
  if (result.success && result.data) {
    return result.data;
  }
  return { exists: false };
}

/**
 * Save post to Kurs
 */
async function savePost(postData) {
  const result = await apiRequest('/api/post', 'POST', postData);
  if (result.success && result.data) {
    return result.data;
  }
  return { success: false };
}

/**
 * Create feed save button
 */
function createFeedButton(postElement, postData, savedStatus) {
  const button = document.createElement('button');
  button.className = FEED_BUTTON_CLASS;

  if (savedStatus.exists) {
    button.innerHTML = '&#10003; Gemt';
    button.classList.add('kurs-feed-saved');
    button.disabled = true;
  } else {
    button.innerHTML = '+ Gem';
    button.addEventListener('click', async (e) => {
      e.preventDefault();
      e.stopPropagation();

      button.disabled = true;
      button.innerHTML = '...';

      const result = await savePost(postData);

      if (result.success) {
        button.innerHTML = '&#10003; Gemt';
        button.classList.add('kurs-feed-saved');
      } else {
        button.innerHTML = '&#10060;';
        button.disabled = false;
        setTimeout(() => {
          button.innerHTML = '+ Gem';
        }, 2000);
      }
    });
  }

  return button;
}

/**
 * Update feed button to saved state
 */
function markFeedButtonAsSaved(button) {
  button.innerHTML = '&#10003; Gemt';
  button.classList.add('kurs-feed-saved');
  button.disabled = true;
  // Remove click listener by cloning
  const newButton = button.cloneNode(true);
  button.parentNode.replaceChild(newButton, button);
}

/**
 * Process a single feed post
 */
async function processFeedPost(postElement) {
  // Skip if already processed
  if (postElement.dataset.kursProcessed) {
    return;
  }
  postElement.dataset.kursProcessed = 'true';

  const postData = scrapePostData(postElement);

  // Skip posts without valid URL or content
  if (!postData.post_url || !postData.content || !postData.author_name) {
    console.log('[Kurs] Skipping post - missing data:', {
      hasUrl: !!postData.post_url,
      hasContent: !!postData.content,
      hasAuthor: !!postData.author_name
    });
    return;
  }

  // Find a place to insert the button (near the actions) - try multiple selectors
  const actionsSelectors = [
    '.feed-shared-social-actions',
    '.social-actions-button',
    '.feed-shared-social-action-bar',
    '[data-test-id="social-actions"]',
    '.social-details-social-counts',
    '.social-details-social-activity',
    '.feed-shared-update-v2__social-actions',
    '.update-components-actions-overflow-menu'
  ];

  let actionsContainer = null;
  for (const selector of actionsSelectors) {
    actionsContainer = postElement.querySelector(selector);
    if (actionsContainer) {
      console.log('[Kurs] Found actions container with:', selector);
      break;
    }
  }

  // Create button immediately (not saved state)
  const button = createFeedButton(postElement, postData, { exists: false });

  if (actionsContainer) {
    // Insert before the actions
    actionsContainer.parentNode.insertBefore(button, actionsContainer);
    console.log('[Kurs] Added button to post (before actions)');
  } else {
    // Fallback: append to the post element itself
    console.log('[Kurs] Could not find actions container, appending to post');
    // Try to find the main content area or just append to post
    const contentArea = postElement.querySelector('.feed-shared-update-v2__content') ||
                        postElement.querySelector('.update-components-text') ||
                        postElement;
    if (contentArea && contentArea.parentNode) {
      contentArea.parentNode.insertBefore(button, contentArea.nextSibling);
      console.log('[Kurs] Added button after content area');
    } else {
      postElement.appendChild(button);
      console.log('[Kurs] Appended button to post element');
    }
  }

  // Check if already saved in the background, then update button if needed
  try {
    const savedStatus = await checkIfPostSaved(postData.post_url);
    if (savedStatus.exists) {
      // Find the button we just added (it should still be in the DOM)
      const buttons = postElement.querySelectorAll(`.${FEED_BUTTON_CLASS}`);
      const ourButton = buttons[buttons.length - 1]; // Get the most recently added one
      if (ourButton && !ourButton.classList.contains('kurs-feed-saved')) {
        markFeedButtonAsSaved(ourButton);
      }
    }
  } catch (e) {
    // Ignore errors - button is already shown
    console.log('[Kurs] Could not check if post is saved:', e);
  }
}

/**
 * Initialize feed scraping
 */
function initFeed() {
  // Find all feed posts - multiple selectors for different layouts
  const feedPostSelectors = [
    '.feed-shared-update-v2',
    '.occludable-update',
    '[data-id*="urn:li:activity"]',
    '.scaffold-finite-scroll__content > div'
  ];

  let feedPosts = [];
  for (const selector of feedPostSelectors) {
    const posts = document.querySelectorAll(selector);
    if (posts.length > 0) {
      feedPosts = posts;
      console.log(`[Kurs] Found ${posts.length} posts with selector: ${selector}`);
      break;
    }
  }

  if (feedPosts.length === 0) {
    console.log('[Kurs] No feed posts found');
    return;
  }

  feedPosts.forEach(post => {
    processFeedPost(post);
  });
}

// ========================================
// PROFILE PAGE FUNCTIONALITY (Create Lead)
// ========================================

const PROFILE_BUTTON_CLASS = 'kurs-profile-btn';

/**
 * Check if we're on a profile page
 */
function isProfilePage() {
  return window.location.pathname.startsWith('/in/');
}

/**
 * Get profile URL from current page
 */
function getProfileUrl() {
  // Clean URL - remove query params and trailing slashes
  const url = window.location.href.split('?')[0].replace(/\/+$/, '');
  return url;
}

/**
 * Scrape profile data from LinkedIn profile page
 */
function scrapeProfileData() {
  // Find main content area (not sticky header)
  const mainContent = document.querySelector('main') || document;

  // Name - search inside main to avoid sticky header
  const nameSelectors = [
    'h1.text-heading-xlarge',
    'h1.inline.t-24',
    '.pv-text-details__left-panel h1',
    '.top-card-layout__title',
    'h1[data-anonymize="person-name"]',
    'main h1'  // Fallback: any h1 inside main
  ];
  let contactName = '';
  for (const selector of nameSelectors) {
    const el = mainContent.querySelector(selector);
    if (el && el.textContent.trim()) {
      contactName = el.textContent.trim();
      console.log('[Kurs] Found name with selector:', selector, '=', contactName);
      break;
    }
  }

  // Headline/Title
  const headlineSelectors = [
    '.text-body-medium.break-words',
    '.pv-text-details__left-panel .text-body-medium',
    '.top-card-layout__headline',
    '[data-anonymize="headline"]'
  ];
  let contactTitle = '';
  for (const selector of headlineSelectors) {
    const el = document.querySelector(selector);
    if (el && el.textContent.trim()) {
      contactTitle = el.textContent.trim();
      break;
    }
  }

  // Current company from headline or experience
  let companyName = '';
  let companyLinkedInUrl = '';

  // Try to extract company from headline (often "Title at Company")
  if (contactTitle && contactTitle.includes(' at ')) {
    const parts = contactTitle.split(' at ');
    if (parts.length > 1) {
      companyName = parts[parts.length - 1].trim();
    }
  }

  // Try experience section for current job
  const experienceSection = document.querySelector('#experience');
  if (experienceSection) {
    const firstJob = experienceSection.closest('section')?.querySelector('li');
    if (firstJob) {
      // Company name
      const companyEl = firstJob.querySelector('.t-bold span[aria-hidden="true"]') ||
                        firstJob.querySelector('.pv-entity__secondary-title');
      if (companyEl && !companyName) {
        companyName = companyEl.textContent.trim().split('·')[0].trim();
      }
      // Company link
      const companyLink = firstJob.querySelector('a[href*="/company/"]');
      if (companyLink) {
        companyLinkedInUrl = companyLink.href.split('?')[0];
      }
    }
  }

  // Summary/About section
  let summary = '';
  const aboutSection = document.querySelector('#about');
  if (aboutSection) {
    const aboutContent = aboutSection.closest('section')?.querySelector('.pv-shared-text-with-see-more span[aria-hidden="true"]') ||
                         aboutSection.closest('section')?.querySelector('.inline-show-more-text');
    if (aboutContent) {
      summary = aboutContent.textContent.trim();
    }
  }

  // Experience list
  const experience = [];
  const experienceItems = document.querySelectorAll('#experience ~ .pvs-list__outer-container li.artdeco-list__item');
  experienceItems.forEach((item, index) => {
    if (index >= 5) return; // Max 5 jobs

    const titleEl = item.querySelector('.t-bold span[aria-hidden="true"]');
    const companyEl = item.querySelector('.t-normal span[aria-hidden="true"]');
    const durationEl = item.querySelector('.t-normal.t-black--light span[aria-hidden="true"]');

    if (titleEl) {
      experience.push({
        title: titleEl.textContent.trim(),
        company: companyEl ? companyEl.textContent.trim().split('·')[0].trim() : '',
        duration: durationEl ? durationEl.textContent.trim() : ''
      });
    }
  });

  // Recent posts (Activity section)
  const recentPosts = [];
  const activitySection = document.querySelector('.pv-recent-activity-section') ||
                          document.querySelector('[data-view-name="profile-component-entity"]');

  // Try to find posts in activity
  const postElements = document.querySelectorAll('.feed-shared-update-v2, .pv-recent-activity-detail');
  postElements.forEach((post, index) => {
    if (index >= 10) return; // Max 10 posts

    const contentEl = post.querySelector('.feed-shared-update-v2__description, .pv-recent-activity-detail__text');
    const dateEl = post.querySelector('.feed-shared-actor__sub-description, .pv-recent-activity-detail__date');

    if (contentEl && contentEl.textContent.trim()) {
      recentPosts.push({
        content: contentEl.textContent.trim(),
        date: dateEl ? dateEl.textContent.trim() : ''
      });
    }
  });

  // Build person research object
  const personResearch = {
    headline: contactTitle,
    summary: summary,
    experience: experience,
    recent_posts: recentPosts,
    fetchedAt: new Date().toISOString()
  };

  console.log('[Kurs] Scraped profile:', { contactName, contactTitle, companyName, personResearch });

  return {
    contact_name: contactName || 'Ukendt',
    contact_title: contactTitle || null,
    contact_linkedin_url: getProfileUrl(),
    company_name: companyName || 'Ukendt virksomhed',
    company_linkedin_url: companyLinkedInUrl || null,
    person_research: personResearch
  };
}

/**
 * Check if lead already exists
 */
async function checkIfLeadSaved(url) {
  const result = await apiRequest(`/api/lead/exists?url=${encodeURIComponent(url)}`);
  if (result.success && result.data) {
    return result.data.exists;
  }
  return false;
}

/**
 * Save lead to Kurs
 */
async function saveLead(leadData) {
  const result = await apiRequest('/api/lead', 'POST', leadData);
  return result;
}

/**
 * Create profile save button
 */
function createProfileButton(alreadySaved = false) {
  const button = document.createElement('button');
  button.className = PROFILE_BUTTON_CLASS;

  if (alreadySaved) {
    button.innerHTML = '<span class="kurs-icon">&#10003;</span> Lead gemt';
    button.classList.add('kurs-already-saved');
    button.disabled = true;
  } else {
    button.innerHTML = '<span class="kurs-icon">&#128100;</span> Opret Lead';
    button.addEventListener('click', handleProfileSaveClick);
  }

  return button;
}

/**
 * Handle profile save button click
 */
async function handleProfileSaveClick(event) {
  event.preventDefault();
  event.stopPropagation();

  const button = event.currentTarget;

  if (button.disabled) return;

  button.disabled = true;
  button.innerHTML = '<span class="kurs-icon">&#8987;</span> Gemmer...';
  button.classList.add('kurs-saving');

  try {
    const data = scrapeProfileData();
    const result = await saveLead(data);

    if (result.success && result.data) {
      if (result.data.already_existed) {
        button.innerHTML = '<span class="kurs-icon">&#10003;</span> Lead opdateret';
      } else {
        button.innerHTML = '<span class="kurs-icon">&#10003;</span> Lead gemt!';
      }
      button.classList.remove('kurs-saving');
      button.classList.add('kurs-success');

      setTimeout(() => {
        button.innerHTML = '<span class="kurs-icon">&#10003;</span> Lead gemt';
        button.classList.remove('kurs-success');
        button.classList.add('kurs-already-saved');
      }, 2000);
    } else {
      throw new Error(result.error || 'Unknown error');
    }

  } catch (error) {
    console.error('[Kurs] Error saving lead:', error);
    button.innerHTML = '<span class="kurs-icon">&#10060;</span> Kurs kører ikke';
    button.classList.remove('kurs-saving');
    button.classList.add('kurs-error');
    button.disabled = false;

    setTimeout(() => {
      button.innerHTML = '<span class="kurs-icon">&#128100;</span> Opret Lead';
      button.classList.remove('kurs-error');
    }, 3000);
  }
}

/**
 * Check if we're on an actual profile page (not search, etc.)
 */
function isActualProfilePage() {
  // Must start with /in/username - allow subpages like /in/username/details/
  // But not search results or other non-profile pages
  const path = window.location.pathname;
  // Match /in/ followed by a username (letters, numbers, dashes, underscores, and special chars like ø, æ, å)
  const isProfile = /^\/in\/[^\/]+/.test(path);
  console.log('[Kurs] isActualProfilePage check:', path, '=', isProfile);
  return isProfile;
}

/**
 * Find container for profile button - specifically on the MAIN profile card (not sticky header)
 */
function findProfileButtonContainer() {
  // Only run on actual profile pages (not search results, etc.)
  if (!isActualProfilePage()) {
    console.log('[Kurs] Not an actual profile page, skipping. Path:', window.location.pathname);
    return null;
  }

  // The main profile card is inside the <main> element
  // The sticky header is outside <main>, so we specifically look inside <main>
  const mainContent = document.querySelector('main');
  if (!mainContent) {
    console.log('[Kurs] Could not find main content area');
    return null;
  }

  // Find action buttons container INSIDE main (the profile card, not sticky header)
  // Look for the container with Message/Connect buttons
  const actionsContainer = mainContent.querySelector('.pvs-profile-actions') ||
                          mainContent.querySelector('.pv-top-card-v2-ctas') ||
                          mainContent.querySelector('.ph5 .pvs-profile-actions');

  if (actionsContainer) {
    console.log('[Kurs] Found profile actions container in main content');
    return actionsContainer;
  }

  // Alternative: Find Message button specifically inside main
  const messageBtn = mainContent.querySelector('button[aria-label*="Message"]') ||
                     mainContent.querySelector('button[aria-label*="Besked"]');

  if (messageBtn && messageBtn.parentElement) {
    console.log('[Kurs] Found Message button in main content');
    return messageBtn.parentElement;
  }

  console.log('[Kurs] Could not find profile button container in main');
  return null;
}

/**
 * Update profile button to saved state
 */
function markProfileButtonAsSaved(button) {
  button.innerHTML = '<span class="kurs-icon">&#10003;</span> Lead gemt';
  button.classList.add('kurs-already-saved');
  button.disabled = true;
  // Remove click listener by cloning
  const newButton = button.cloneNode(true);
  button.parentNode.replaceChild(newButton, button);
}

/**
 * Initialize profile page
 */
async function initProfilePage() {
  console.log('[Kurs] Initializing profile page...');

  // Only run on actual profile pages
  if (!isActualProfilePage()) {
    console.log('[Kurs] Not an actual profile page, skipping');
    return;
  }

  // Don't add button if it already exists
  if (document.querySelector(`.${PROFILE_BUTTON_CLASS}`)) {
    console.log('[Kurs] Button already exists, skipping');
    return;
  }

  const container = findProfileButtonContainer();
  if (!container) {
    console.log('[Kurs] Could not find profile container - will retry');
    // Retry after a delay in case page is still loading
    setTimeout(initProfilePage, 2000);
    return;
  }

  // Create button immediately (not saved state)
  const button = createProfileButton(false);

  // Insert button at the end of the actions container (next to Message/More buttons)
  container.appendChild(button);
  console.log('[Kurs] Profile button added');

  // Check if already saved in the background, then update button if needed
  const profileUrl = getProfileUrl();
  try {
    console.log('[Kurs] Checking if lead exists:', profileUrl);
    const alreadySaved = await checkIfLeadSaved(profileUrl);
    console.log('[Kurs] Lead already saved:', alreadySaved);
    if (alreadySaved) {
      const existingButton = document.querySelector(`.${PROFILE_BUTTON_CLASS}`);
      if (existingButton) {
        markProfileButtonAsSaved(existingButton);
      }
    }
  } catch (e) {
    // Ignore errors - button is already shown
    console.log('[Kurs] Could not check if lead is saved:', e);
  }
}

/**
 * Clean up profile buttons
 */
function cleanupProfileButtons() {
  const oldButtons = document.querySelectorAll(`.${PROFILE_BUTTON_CLASS}`);
  oldButtons.forEach(btn => btn.remove());
}

// ========================================
// MAIN INITIALIZATION
// ========================================

// Use MutationObserver to detect SPA navigation and new content
const observer = new MutationObserver((mutations) => {
  const currentPath = window.location.pathname;

  // Job pages
  if (currentPath.includes('/jobs/')) {
    // Look for significant DOM changes that might indicate new job loaded
    for (const mutation of mutations) {
      if (mutation.addedNodes.length > 0) {
        // Debounce the init call
        clearTimeout(window.kursJobTimeout);
        window.kursJobTimeout = setTimeout(() => {
          initJobPage();
        }, 500);
        break;
      }
    }
  }

  // Feed page
  if (currentPath === '/feed/' || currentPath === '/feed') {
    for (const mutation of mutations) {
      if (mutation.addedNodes.length > 0) {
        // Debounce feed processing
        clearTimeout(window.kursFeedTimeout);
        window.kursFeedTimeout = setTimeout(() => {
          initFeed();
        }, 300);
        break;
      }
    }
  }

  // Profile pages
  if (currentPath.startsWith('/in/')) {
    for (const mutation of mutations) {
      if (mutation.addedNodes.length > 0) {
        clearTimeout(window.kursProfileTimeout);
        window.kursProfileTimeout = setTimeout(() => {
          initProfilePage();
        }, 500);
        break;
      }
    }
  }
});

// Start observing
observer.observe(document.body, {
  childList: true,
  subtree: true
});

// Initial run with delay to ensure page is loaded
setTimeout(() => {
  const currentPath = window.location.pathname;
  if (currentPath.includes('/jobs/')) {
    initJobPage();
  } else if (currentPath === '/feed/' || currentPath === '/feed') {
    initFeed();
  } else if (currentPath.startsWith('/in/')) {
    initProfilePage();
  }
}, 1000);

// Also run on popstate for back/forward navigation
window.addEventListener('popstate', () => {
  cleanupJobButtons();
  cleanupFeedButtons();
  cleanupProfileButtons();
  lastJobId = ''; // Reset job ID tracking
  setTimeout(() => {
    const currentPath = window.location.pathname;
    if (currentPath.includes('/jobs/')) {
      initJobPage();
    } else if (currentPath === '/feed/' || currentPath === '/feed') {
      initFeed();
    } else if (currentPath.startsWith('/in/')) {
      initProfilePage();
    }
  }, 1000);
});

// Detect job ID changes for LinkedIn's SPA navigation (clicking jobs in list)
let lastCheckedJobId = getJobIdFromUrl();
setInterval(() => {
  const currentJobId = getJobIdFromUrl();

  if (currentJobId && currentJobId !== lastCheckedJobId) {
    lastCheckedJobId = currentJobId;

    // Job ID changed - re-init to check new job
    clearTimeout(window.kursJobTimeout);
    window.kursJobTimeout = setTimeout(() => {
      initJobPage();
    }, 300);
  }
}, 500);
